import { Injectable } from '@nestjs/common';
import { InjectRepository } from '@nestjs/typeorm';
import { Like, Repository } from 'typeorm';
import { Blog } from './blog.entity';
import { CreateBlogDto } from './dto/create-blog.dto';
import { UpdateBlogDto } from './dto/update-blog.dto';
import { BlogCategory } from 'src/blog_category/entity/blog_category.entity';

@Injectable()
export class BlogService {
  constructor(
    @InjectRepository(Blog)
    private readonly blogRepo: Repository<Blog>,

    @InjectRepository(BlogCategory)
    private readonly categoryRepo: Repository<BlogCategory>,
  ) {}

  // ✅ Create Blog
  async create(
    createDto: CreateBlogDto,
    image?: Express.Multer.File,
    createdBy?: number,
  ) {
    try {
      const blog = this.blogRepo.create({
        ...createDto,
        created_by: createdBy,
      });

      // Handle optional category
      if (createDto.category_id) {
        const category = await this.categoryRepo.findOne({
          where: { id: createDto.category_id },
        });
        if (category) blog.category = category;
      }
      const checkslog = await this.blogRepo.findOne({
        where: { slug: blog.slug },
      });
      // return checkslog;
      if (checkslog != null) {
        throw new Error('Slug already exists');
      }

      if (image) blog.image = image.filename;

      const saved = await this.blogRepo.save(blog);
      return {
        status: true,
        message: 'Blog created successfully',
        data: saved,
      };
    } catch (error) {
      return {
        status: false,
        message: 'Failed to create blog',
        data: error.message,
      };
    }
  }

  // ✅ Find all blogs (with pagination & optional filters)
  async findAll(status?: number, page = 1, limit = 10, search?: string) {
    try {
      const skip = (page - 1) * limit;

      const whereConditions: any = {};
      if (status !== undefined) whereConditions.status = status;
      if (search) whereConditions.title = Like(`%${search}%`);

      const [data, total] = await this.blogRepo.findAndCount({
        where: whereConditions,
        skip,
        take: limit,
        order: { id: 'DESC' },
        relations: ['category'], // include category info
      });

      return {
        status: true,
        message: 'Blogs fetched successfully',
        data,
        pagination: {
          total,
          currentPage: page,
          totalPages: Math.ceil(total / limit),
          limit,
        },
      };
    } catch (error) {
      return {
        status: false,
        message: 'Failed to fetch blogs',
        data: error.message,
      };
    }
  }

  // ✅ Find one by ID
  async findOne(id: number) {
    try {
      const blog = await this.blogRepo.findOne({
        where: { id },
        relations: ['category'],
      });

      if (!blog) {
        return { status: false, message: `Blog with ID ${id} not found` };
      }

      // ✅ Explicitly typed array
      let relatedBlogs: Blog[] = [];

      if (blog.category?.id) {
        relatedBlogs = await this.blogRepo.find({
          where: {
            category: { id: blog.category.id },
            status: 1,
          },
          relations: ['category'],
          take: 5,
        });
      }

      const filteredRelatedBlogs = relatedBlogs.filter((b) => b.id !== blog.id);

      return {
        status: true,
        message: 'Blog fetched successfully',
        data: {
          ...blog,
          related_blogs: filteredRelatedBlogs,
        },
      };
    } catch (error) {
      return {
        status: false,
        message: 'Failed to fetch blog',
        data: error.message,
      };
    }
  }

  // ✅ Find one by slug
  async findBySlug(slug: string) {
    try {
      const blog = await this.blogRepo.findOne({
        where: { slug },
        relations: ['category'],
      });
      if (!blog) {
        return { status: false, message: `Blog with slug "${slug}" not found` };
      }
      return { status: true, message: 'Blog fetched successfully', data: blog };
    } catch (error) {
      return {
        status: false,
        message: 'Failed to fetch blog',
        data: error.message,
      };
    }
  }

  // ✅ Update Blog
  async update(
    id: number,
    updateDto: UpdateBlogDto,
    image?: Express.Multer.File,
    updatedBy?: number,
  ) {
    try {
      const existing = await this.blogRepo.findOne({
        where: { id },
        relations: ['category'],
      });
      if (!existing) {
        return { status: false, message: `Blog with ID ${id} not found` };
      }

      Object.assign(existing, updateDto, { updated_by: updatedBy });

      // Handle optional category update
      if (updateDto.category_id) {
        const category = await this.categoryRepo.findOne({
          where: { id: updateDto.category_id },
        });
        if (category) existing.category = category;
      } else {
        existing.category = null; // remove category if not provided
      }

      if (image) existing.image = image.filename;

      const updated = await this.blogRepo.save(existing);
      return {
        status: true,
        message: 'Blog updated successfully',
        data: updated,
      };
    } catch (error) {
      return {
        status: false,
        message: 'Failed to update blog',
        data: error.message,
      };
    }
  }

  // ✅ Toggle status
  async toggleStatus(id: number) {
    try {
      const blog = await this.blogRepo.findOne({ where: { id } });
      if (!blog) {
        return { status: false, message: `Blog with ID ${id} not found` };
      }

      blog.status = blog.status === 1 ? 0 : 1;
      const updated = await this.blogRepo.save(blog);

      return {
        status: true,
        message: `Blog status updated to ${blog.status === 1 ? 'Active' : 'Inactive'}`,
        data: updated,
      };
    } catch (error) {
      return {
        status: false,
        message: 'Failed to toggle blog status',
        data: error.message,
      };
    }
  }
}
