import {
  Controller,
  Get,
  Post,
  Body,
  Param,
  Patch,
  Delete,
  ParseIntPipe,
  UseGuards,
} from '@nestjs/common';
import { BlogCategoryService } from './blog_category.service';
import { CreateCategoryDto } from 'src/category/dto/create-category.dto';
import { UpdateCategoryDto } from 'src/category/dto/update-category.dto';
import { AdminJwtAuthGuard } from 'src/auth/admin/admin-jwt.guard';

@Controller('admin/blog/categories')
@UseGuards(AdminJwtAuthGuard)
export class BlogCategoryController {
  constructor(private readonly categoryService: BlogCategoryService) {}

  // ✅ Create a new category
  @Post('store')
  async create(@Body() createCategoryDto: CreateCategoryDto) {
    const category = await this.categoryService.create(createCategoryDto);
    return {
      status: true,
      message: 'Category created successfully',
      data: category,
    };
  }

  // ✅ Get all categories
  @Get()
  async findAll() {
    const categories = await this.categoryService.findAll();
    return {
      status: true,
      message: 'Categories fetched successfully',
      total: categories.length,
      data: categories,
    };
  }

  // ✅ Get single category by ID
  @Get(':id')
  async findOne(@Param('id', ParseIntPipe) id: number) {
    const category = await this.categoryService.findOne(id);
    return {
      status: true,
      message: 'Category fetched successfully',
      data: category,
    };
  }

  // ✅ Update category by ID
  @Patch(':id')
  async update(
    @Param('id', ParseIntPipe) id: number,
    @Body() updateCategoryDto: UpdateCategoryDto,
  ) {
    const updated = await this.categoryService.update(id, updateCategoryDto);
    return {
      status: true,
      message: 'Category updated successfully',
      data: updated,
    };
  }

  // ✅ Delete category by ID
  @Delete(':id')
  async remove(@Param('id', ParseIntPipe) id: number) {
    const deleted = await this.categoryService.remove(id);
    return {
      status: true,
      message: 'Category deleted successfully',
      data: deleted,
    };
  }
}
