import { BadRequestException } from '@nestjs/common';
import { diskStorage } from 'multer';
import { extname, resolve } from 'path';
import * as fs from 'fs';

export const multerConfig = (destination: string) => ({
  storage: diskStorage({
    destination: (req, file, cb) => {
      const uploadPath = resolve(process.cwd(), destination);
      if (!fs.existsSync(uploadPath)) {
        fs.mkdirSync(uploadPath, { recursive: true });
      }
      cb(null, uploadPath);
    },
    filename: (req, file, cb) => {
      const ext = extname(file.originalname);

      // 🧹 Clean up fieldname (e.g., "imageGroups[0][]" → "")
      const cleanFieldname = file.fieldname.replace(/imageGroups\[\d+\]\[\]/, '').trim();

      // 🆕 Generate a unique name
      const uniqueName = `${Date.now()}${ext}`;

      // ✅ Final filename (no messy prefixes)
      cb(null, uniqueName);
    },
  }),

  fileFilter: (req, file, cb) => {
    const allowedMimeTypes = ['image/jpeg', 'image/png', 'image/jpg', 'image/gif','image/svg+xml',
      'image/webp','image/x-icon',];
    if (allowedMimeTypes.includes(file.mimetype)) {
      cb(null, true);
    } else {
      cb(new BadRequestException('Only image files(jpeg, png, jpg, gif, svg, webp, x-icon) are allowed!'), false);
    }
  },
});
