import { Injectable } from '@nestjs/common';
import { InjectRepository } from '@nestjs/typeorm';
import { Not, Repository } from 'typeorm';
import { Gallery } from './gallery.entity';

@Injectable()
export class GalleryService {
    constructor(
        @InjectRepository(Gallery)
        private readonly galleryRepo: Repository<Gallery>,
    ) { }

    async create(dto: any, file?: Express.Multer.File) {
        try {
            const gallery = new Gallery();
            gallery.title = dto.title || null;
            gallery.alt_image = dto.alt_image || null;
            gallery.status = 1;

            // handle image upload
            if (file) {
                gallery.images = file.filename;
                gallery.Image_path = `galleries/${file.filename}`;
            } else {
                gallery.images = '';
                gallery.Image_path = '';
            }

            const saved = await this.galleryRepo.save(gallery);
            return {
                status: true,
                message: 'Gallery created successfully',
                data: saved,
            };
        } catch (err) {
            return { status: false, message: err.message, data: null };
        }
    }

    async findAll() {
        const galleries = await this.galleryRepo.find({
            where: { status: Not(2) }, // exclude deleted
            order: { id: "DESC" }
        });
        return { status: true, data: galleries };
    }

    async findById(id: number) {
        const gallery = await this.galleryRepo.findOne({ where: { id } });
        if (!gallery || gallery.status === 2) {
            return { status: false, message: 'Gallery not found or deleted' };
        }
        return { status: true, data: gallery };
    }

    async update(id: number, dto: any, file?: Express.Multer.File) {
        try {
            const gallery = await this.galleryRepo.findOne({ where: { id } });

            if (!gallery) {
                return { status: false, message: `Gallery ID ${id} not found`, data: null };
            }

            // Update fields if provided
            gallery.title = dto.title ?? gallery.title;
            gallery.alt_image = dto.alt_image ?? gallery.alt_image;

            // If a new file is uploaded, replace old image info
            if (file) {
                gallery.images = file.filename;
                gallery.Image_path = `galleries/${file.filename}`;
            }

            const updated = await this.galleryRepo.save(gallery);

            return {
                status: true,
                message: 'Gallery updated successfully',
                data: updated,
            };
        } catch (err) {
            return { status: false, message: err.message, data: null };
        }
    }

    async softDelete(id: number) {
        const gallery = await this.galleryRepo.findOne({ where: { id } });
        if (!gallery) {
            return { status: false, message: 'Gallery not found' };
        }

        if (gallery.status === 2) {
            return { status: false, message: 'Already deleted' };
        }

        gallery.status = 2;
        await this.galleryRepo.save(gallery);
        return { status: true, message: 'Gallery soft deleted' };
    }

    async toggleStatus(id: number) {
        const gallery = await this.galleryRepo.findOne({ where: { id } });
        if (!gallery || gallery.status === 2) {
            return { status: false, message: 'Gallery not found or deleted' };
        }

        gallery.status = gallery.status === 1 ? 0 : 1;
        await this.galleryRepo.save(gallery);
        return {
            status: true,
            message: `Gallery ${gallery.status === 1 ? 'activated' : 'deactivated'}`,
            data: gallery,
        };
    }
}
