import {
    Controller,
    Get,
    Post,
    Put,
    Delete,
    Param,
    Body,
    ParseIntPipe,
    UseInterceptors,
    UploadedFile,
    UseGuards,
    Req,
} from '@nestjs/common';
import { NewsService } from './news.service';
import { CreateNewsDto } from './dto/create-news.dto';
import { UpdateNewsDto } from './dto/update-news.dto';
import { FileInterceptor } from '@nestjs/platform-express';
import { multerConfig } from 'src/common/utils/multer.config';
import { AdminJwtAuthGuard } from 'src/auth/admin/admin-jwt.guard';

@Controller('news')
export class NewsController {
    constructor(private readonly newsService: NewsService) { }

    @Post('create')
    @UseGuards(AdminJwtAuthGuard)
    @UseInterceptors(FileInterceptor('image', multerConfig('uploads/news')))
    async create(
        @Body() dto: CreateNewsDto,
        @UploadedFile() file?: Express.Multer.File,
        @Req() req?: any
    ) {
        if (file) dto.image = file.filename;
        const admin = req.user as any;
        const createdBy = admin?.id;
        return await this.newsService.create(dto, createdBy);
    }

    @Get('list')
    async findAll() {
        return await this.newsService.findAll();
    }

    @Get('findby/:id')
    async findOne(@Param('id', ParseIntPipe) id: number) {
        return await this.newsService.findOne(id);
    }

    @Put('updateby/:id')
    @UseGuards(AdminJwtAuthGuard)
    @UseInterceptors(FileInterceptor('image', multerConfig('uploads/news')))
    async update(
        @Param('id', ParseIntPipe) id: number,
        @Body() dto: UpdateNewsDto,
        @UploadedFile() file?: Express.Multer.File,
        @Req() req?: any
    ) {
        if (file) dto.image = file.filename;
        const admin = req.user as any;
        const updatedBy = admin?.id;
        return await this.newsService.update(id, dto, updatedBy);
    }

    @Put('toggleStatus/:id')
    @UseGuards(AdminJwtAuthGuard)
    async toggleStatus(@Param('id', ParseIntPipe) id: number) {
        return await this.newsService.toggleStatus(id);
    }
}
