import {
    Injectable
} from '@nestjs/common';
import { InjectRepository } from '@nestjs/typeorm';
import { Repository } from 'typeorm';
import { News } from './news.entity';
import { CreateNewsDto } from './dto/create-news.dto';
import { UpdateNewsDto } from './dto/update-news.dto';

@Injectable()
export class NewsService {
    constructor(
        @InjectRepository(News)
        private readonly newsRepo: Repository<News>,
    ) { }

    async create(dto: CreateNewsDto, createdBy?: number) {
        try {
            const news = this.newsRepo.create({ ...dto, created_by: createdBy });
            const saved = await this.newsRepo.save(news);

            return {
                status: true,
                message: 'News created successfully',
                data: saved,
            };
        } catch (error) {
            return {
                status: false,
                message: 'Failed to create news',
                data: error.message,
            };
        }
    }

    async findAll() {
        try {
            const allNews = await this.newsRepo.find({where: {}, order: { id: 'DESC' }});
            return {
                status: true,
                message: 'News fetched successfully',
                data: allNews,
            };
        } catch (error) {
            return {
                status: false,
                message: 'Failed to fetch news',
                data: error.message,
            };
        }
    }

    async findOne(id: number) {
        try {
            const news = await this.newsRepo.findOne({ where: { id } });
            if (!news) {
                return {
                    status: false,
                    message: `News with ID ${id} not found`,
                };
            }
            return {
                status: true,
                message: 'News fetched successfully',
                data: news,
            };
        } catch (error) {
            return {
                status: false,
                message: 'Failed to fetch news',
                data: error.message,
            };
        }
    }

    async update(id: number, dto: UpdateNewsDto, updatedBy?: number) {
        try {
            const existing = await this.newsRepo.findOne({ where: { id } });
            if (!existing) {
                return {
                    status: false,
                    message: `News with ID ${id} not found`,
                };
            }

            Object.assign(existing, dto, { updated_by: updatedBy });
            const updated = await this.newsRepo.save(existing);

            return {
                status: true,
                message: 'News updated successfully',
                data: updated,
            };
        } catch (error) {
            return {
                status: false,
                message: 'Failed to update news',
                data: error.message,
            };
        }
    }

    async toggleStatus(id: number) {
        try {
            const news = await this.newsRepo.findOne({ where: { id } });
            if (!news) {
                return {
                    status: false,
                    message: `News with ID ${id} not found`,
                };
            }

            news.status = news.status === 1 ? 0 : 1;
            const updated = await this.newsRepo.save(news);

            return {
                status: true,
                message: `News status updated to ${news.status === 1 ? 'Active' : 'Inactive'}`,
                data: updated,
            };
        } catch (error) {
            return {
                status: false,
                message: 'Failed to toggle news status',
                data: error.message,
            };
        }
    }
}
