import { Controller, Get, Post, Put, Patch, Param, Body, UploadedFile, UseInterceptors, UseGuards, Req } from '@nestjs/common';
import { FileInterceptor } from '@nestjs/platform-express';
import { ProgramDetailService } from './program-detail.service';
import { CreateProgramDetailDto } from './dto/create-program-detail.dto';
import { UpdateProgramDetailDto } from './dto/update-program-detail.dto';
import { multerConfig } from 'src/common/utils/multer.config'; 
import { AdminJwtAuthGuard } from 'src/auth/admin/admin-jwt.guard';

@Controller('program-details')
export class ProgramDetailController {
    constructor(private readonly service: ProgramDetailService) { }

    @Post('create')
    @UseGuards(AdminJwtAuthGuard)
    @UseInterceptors(FileInterceptor('image', multerConfig('uploads')))
    create(@Body() dto: CreateProgramDetailDto, @UploadedFile() file: Express.Multer.File, @Req() req: any) {
        const admin = req.user as any;
        const createdBy = admin?.id;
        return this.service.create(dto, file?.filename, createdBy);
    }

    @Get('list')
    findAll() {
        return this.service.findAll();
    }

    @Get('findby/:id')
    findOne(@Param('id') id: number) {
        return this.service.findOne(+id);
    }

    @Put('updateby/:id')
    @UseGuards(AdminJwtAuthGuard)
    @UseInterceptors(FileInterceptor('image', multerConfig('uploads')))
    update(
        @Param('id') id: number,
        @Body() dto: UpdateProgramDetailDto,
        @UploadedFile() file: Express.Multer.File,
        @Req() req: any
    ) {
        const admin = req.user as any;
        const updatedBy = admin?.id;
        return this.service.update(+id, dto, file?.filename, updatedBy);
    }

    @Put('toggle-status/:id')
    @UseGuards(AdminJwtAuthGuard)
    toggleStatus(@Param('id') id: number) {
        return this.service.toggleStatus(+id);
    }
}
