import { Injectable } from '@nestjs/common';
import { InjectRepository } from '@nestjs/typeorm';
import { Repository } from 'typeorm';
import { ProgramDetail } from './program-detail.entity';
import { CreateProgramDetailDto } from './dto/create-program-detail.dto';
import { UpdateProgramDetailDto } from './dto/update-program-detail.dto';
import { Program } from 'src/program/program.entity';

@Injectable()
export class ProgramDetailService {
    constructor(
        @InjectRepository(ProgramDetail)
        private readonly repo: Repository<ProgramDetail>,
        @InjectRepository(Program)
        private readonly programRepo: Repository<Program>,
    ) { }

    async create(dto: CreateProgramDetailDto, filename?: string, createdBy?: number) {
        try {
            const program = await this.programRepo.findOne({
                where: { id: dto.program_id, status: 1 },
            });

            if (!program) {
                return {
                    status: false,
                    message: `Program with ID ${dto.program_id} not found`,
                    data: null,
                };
            }

            const detail = this.repo.create({
                ...dto,
                image: filename,
                created_by: createdBy,
            });
            const saved = await this.repo.save(detail);

            return {
                status: true,
                message: 'Program detail created successfully',
                data: saved,
            };
        } catch (error) {
            return {
                status: false,
                message: error.message || 'Failed to create program detail',
                data: null,
            };
        }
    }

    async findAll() {
        try {
            const data = await this.repo.find({
                where: { status: 1 },
                relations: ['program'],
                order: { id: 'DESC' },
            });

            return {
                status: true,
                message: 'Program details fetched successfully',
                data,
            };
        } catch (error) {
            return {
                status: false,
                message: 'Failed to fetch program details',
                data: null,
            };
        }
    }

    async findOne(id: number) {
        try {
            const detail = await this.repo.findOne({
                where: { id },
                relations: ['program'],
            });

            if (!detail) {
                return {
                    status: false,
                    message: 'Program detail not found',
                    data: null,
                };
            }

            return {
                status: true,
                message: 'Program detail fetched successfully',
                data: detail,
            };
        } catch (error) {
            return {
                status: false,
                message: 'Failed to fetch program detail',
                data: null,
            };
        }
    }

    async update(id: number, dto: UpdateProgramDetailDto, filename?: string, updatedBy?: number) {
        try {
            const existing = await this.repo.findOne({ where: { id } });
            if (!existing) {
                return {
                    status: false,
                    message: 'Program detail not found',
                    data: null,
                };
            }

            Object.assign(existing, dto, { updated_by: updatedBy });
            if (filename) existing.image = filename;
            const updated = await this.repo.save(existing);

            return {
                status: true,
                message: 'Program detail updated successfully',
                data: updated,
            };
        } catch (error) {
            return {
                status: false,
                message: 'Failed to update program detail',
                data: null,
            };
        }
    }

    async toggleStatus(id: number) {
        try {
            const existing = await this.repo.findOne({ where: { id } });
            if (!existing) {
                return {
                    status: false,
                    message: 'Program detail not found',
                    data: null,
                };
            }

            existing.status = existing.status === 1 ? 0 : 1;
            const updated = await this.repo.save(existing);

            return {
                status: true,
                message: `Program detail ${updated.status === 1 ? 'activated' : 'deactivated'} successfully`,
                data: updated,
            };
        } catch (error) {
            return {
                status: false,
                message: 'Failed to toggle program detail status',
                data: null,
            };
        }
    }
}
