import {
    Controller,
    Post,
    Get,
    Put,
    Param,
    Body,
    ParseIntPipe,
    UploadedFiles,
    UseGuards,
    UseInterceptors,
    Delete,
    Req,
    Query,
} from '@nestjs/common';
import { ProgramService } from './program.service';
import { CreateProgramDto } from './dto/create-program.dto';
import { UpdateProgramDto } from './dto/update-program.dto';
import { FilesInterceptor } from '@nestjs/platform-express';
import { multerConfig } from 'src/common/utils/multer.config';
import { AdminJwtAuthGuard } from 'src/auth/admin/admin-jwt.guard';

@Controller('programs')
export class ProgramController {
    constructor(private readonly programService: ProgramService) { }


    @Post('create')
    @UseGuards(AdminJwtAuthGuard)
    @UseInterceptors(FilesInterceptor('images', 10, multerConfig('uploads/programs')))
    async create(
        @Body() dto: CreateProgramDto,
        @UploadedFiles() images: Express.Multer.File[],
        @Req() req: any,
    ) {
        const createdBy = req.user?.id;
        if (dto.details && typeof dto.details === 'string') {
            dto.details = JSON.parse(dto.details);
        }
        return await this.programService.create(dto, createdBy, images);
    }


    @Get('list')
    async findAll(
        @Query('status') status?: number,
        @Query('limit') limit?: number,
        @Query('offset') offset?: number,
    ) {
        return await this.programService.findAll(status, limit, offset);
    }



    @Get('findby/:id')
    async findById(@Param('id', ParseIntPipe) id: number) {
        return await this.programService.findById(id);
    }


    @Put('updateby/:id')
    @UseGuards(AdminJwtAuthGuard)
    @UseInterceptors(FilesInterceptor('images', 10, multerConfig('uploads/programs')))
    async update(
        @Param('id', ParseIntPipe) id: number,
        @Body() dto: UpdateProgramDto,
        @UploadedFiles() images: Express.Multer.File[],
        @Req() req: any,
    ) {
        const updatedBy = req.user?.id;
        return await this.programService.update(id, dto, updatedBy, images);
    }


    @Put('toggleStatus/:id')
    async toggleStatus(@Param('id', ParseIntPipe) id: number) {
        return await this.programService.toggleStatus(id);
    }

    @Delete('deleteby/:id')
    @UseGuards(AdminJwtAuthGuard)
    async softDelete(@Param('id') id: number) {
        return await this.programService.softDelete(id);
    }


}
