import { Injectable } from '@nestjs/common';
import { InjectRepository } from '@nestjs/typeorm';
import { Repository, DataSource, Not } from 'typeorm';
import { Program } from './program.entity';
import { ProgramDetail } from 'src/program-detail/program-detail.entity';
import { CreateProgramDto } from './dto/create-program.dto';
import { UpdateProgramDto } from './dto/update-program.dto';
import { pageHeadings, PageType } from 'src/service/page-heading.entity';

@Injectable()
export class ProgramService {
  constructor(
    @InjectRepository(Program)
    private readonly programRepo: Repository<Program>,

    @InjectRepository(ProgramDetail)
    private readonly programDetailRepo: Repository<ProgramDetail>,

    @InjectRepository(pageHeadings)
    private readonly pageHeading: Repository<pageHeadings>,

    private readonly dataSource: DataSource,
  ) {}

  async create(
    dto: CreateProgramDto,
    createdBy: number,
    images: Express.Multer.File[],
  ) {
    const queryRunner = this.dataSource.createQueryRunner();
    await queryRunner.connect();
    await queryRunner.startTransaction();

    try {
      const mainImage =
        images && images.length > 0 ? images[0].filename : undefined;

      const program = queryRunner.manager.create(Program, {
        // heading: dto.heading,
        title: dto.title,
        description: dto.description,
        // slug: dto.slug || '',
        image: 'programs/' + mainImage,
        date: dto.date,
        location: dto.location,
        campain_status: dto.campain_status,
        required_total_amount: dto.required_total_amount,
        created_by: createdBy,
      });

      const savedProgram = await queryRunner.manager.save(program);

      let detailsToSave: ProgramDetail[] = [];

      // if (dto.details && dto.details.length > 0) {
      //     detailsToSave = dto.details.map((detail, index) => {
      //         const imageFile = images && images[index] ? images[index].filename : undefined;

      //         return queryRunner.manager.create(ProgramDetail, {
      //             program_id: savedProgram.id,
      //             title: detail.title,
      //             description: detail.description,
      //             image: imageFile, // <-- mapped image filename
      //             alter_image: detail.alter_image,
      //             image_name: 'programs/' + detail.image_name,
      //             created_by: createdBy,
      //             status: 1,
      //         });
      //     });

      //     await queryRunner.manager.save(detailsToSave);
      // }

      await queryRunner.commitTransaction();

      return {
        status: true,
        message: 'Program and details inserted successfully',
        program: savedProgram,
        details: detailsToSave || [],
      };
    } catch (error) {
      await queryRunner.rollbackTransaction();
      console.error('Insert error:', error);
      return {
        status: false,
        message: `Failed to insert program and details: ${error.message}`,
      };
    } finally {
      await queryRunner.release();
    }
  }

  //old code
  // async findAll(status?: string, limit?: number) {
  //     try {
  //         const typer = PageType.PROGRAM;
  //         const page_heading = await this.pageHeading.findOne({ where: { type: typer } });

  //         const whereCondition: any = { status: Not(2) };
  //         if (status !== undefined) {
  //             whereCondition.status = status === 'active' ? 1 : 0;
  //         }

  //         // ✅ Query with specific fields + total transaction amount
  //         const query = this.programRepo
  //             .createQueryBuilder('program')
  //             .leftJoin('transaction', 'transaction', 'transaction.support_program = program.id')
  //             .select([
  //                 'program.id AS id',
  //                 'program.title AS title',
  //                 'program.description AS description',
  //                 'program.image AS image',
  //                 'program.meta_title AS meta_title',
  //                 'program.meta_description AS meta_description',
  //                 'program.schema_json AS schema_json',
  //                 'program.canonical_url AS canonical_url',
  //                 'program.keyword AS keyword',
  //                 'program.status AS status',
  //                 'program.date AS date',
  //                 'program.location AS location',
  //                 'program.campain_status AS campain_status',
  //                 'program.required_total_amount AS required_total_amount',
  //                 'program.created_by AS created_by',
  //                 'COALESCE(SUM(transaction.amount), 0) AS total_transaction_amount',
  //             ])
  //             .where(whereCondition)
  //             .groupBy('program.id')
  //             .orderBy('program.id', 'DESC');

  //         // ✅ Apply limit if provided
  //         if (limit && !isNaN(limit)) {
  //             query.take(limit);
  //         }

  //         const result = await query.getRawMany();

  //         return {
  //             status: true,
  //             message: 'Programs fetched successfully',
  //             page_heading,
  //             total: result.length,
  //             data: result.map((p) => ({
  //                 ...p,
  //                 total_transaction_amount: parseFloat(p.total_transaction_amount) || 0,
  //             })),
  //         };
  //     } catch (error) {
  //         return {
  //             status: false,
  //             message: 'Failed to fetch programs',
  //             data: error.message,
  //         };
  //     }
  // }

  // new code
  async findAll(status?: number, limit: number = 10, offset: number = 0) {
    try {
      const typer = PageType.PROGRAM;

      const page_heading = await this.pageHeading.findOne({
        where: { type: typer },
      });

      // ✅ Step 1: Get IDs with pagination
      const idsQuery = this.programRepo
        .createQueryBuilder('program')
        .select('program.id')
        .where('program.status != :deletedStatus', { deletedStatus: 2 })
        // ✅ Conditionally add status filter
        .andWhere(
          status !== undefined && status !== null
            ? 'program.status = :status'
            : '1=1',
          { status },
        )
        .orderBy('program.id', 'DESC')
        .take(limit)
        .skip(offset);

      const idsResult = await idsQuery.getRawMany();
      const ids = idsResult.map((row) => row.program_id);

      if (ids.length === 0) {
        return {
          status: true,
          message: 'Programs fetched successfully',
          page_heading,
          total: 0,
          count: 0,
          data: [],
        };
      }

      // ✅ Step 2: Fetch full data with those IDs
      const query = this.programRepo
        .createQueryBuilder('program')
        .leftJoin(
          'transaction',
          'transaction',
          'transaction.support_program = program.id',
        )
        .select([
          'program.id AS id',
          'program.title AS title',
          'program.description AS description',
          'program.image AS image',
          'program.status AS status',
          'program.date AS date',
          'program.location AS location',
          'program.campain_status AS campain_status',
          'program.required_total_amount AS required_total_amount',
          'program.created_by AS created_by',
          'COALESCE(SUM(transaction.amount), 0) AS total_transaction_amount',
        ])
        .whereInIds(ids)
        .groupBy('program.id')
        .orderBy('program.id', 'DESC');

      const result = await query.getRawMany();

      // ✅ Get total (without filtering by status unless provided)
      const total = await this.programRepo.count({
        where: {
          status: Not(2),
          ...(status !== undefined && status !== null ? { status } : {}),
        },
      });

      return {
        status: true,
        message: 'Programs fetched successfully',
        page_heading,
        total,
        count: result.length,
        data: result.map((p) => ({
          ...p,
          total_transaction_amount: parseFloat(p.total_transaction_amount) || 0,
        })),
      };
    } catch (error) {
      return {
        status: false,
        message: 'Failed to fetch programs',
        data: error.message,
      };
    }
  }

  async findById(id: number) {
    try {
      const program = await this.programRepo.findOne({
        where: { id, status: Not(2) },
        relations: ['program_details'],
      });

      if (!program) {
        return {
          status: false,
          message: `Program with ID ${id} not found`,
          data: null,
        };
      }

      return {
        status: true,
        message: 'Program fetched successfully',
        data: program,
      };
    } catch (error) {
      return {
        status: false,
        message: 'Failed to fetch program',
        data: error.message,
      };
    }
  }

  async update(
    id: number,
    dto: UpdateProgramDto,
    updatedBy: number,
    images?: Express.Multer.File[],
  ) {
    try {
      const existing = await this.programRepo.findOne({
        where: { id },
        relations: ['program_details'],
      });

      if (!existing) {
        return {
          status: false,
          message: `Program with ID ${id} not found`,
        };
      }

      // --- Handle main program image ---
      if (images && images.length > 0) {
        // Use the first uploaded file for main program image
        existing.image = 'programs/' + images[0].filename;
      }

      // Parse details safely
      const parsedDetails =
        typeof dto.details === 'string' ? JSON.parse(dto.details) : dto.details;

      // --- Update existing details or add new ---
      if (parsedDetails && Array.isArray(parsedDetails)) {
        for (let i = 0; i < parsedDetails.length; i++) {
          const detailDto = parsedDetails[i];
          const imageFile = images?.[i];

          // Case 1: Update existing detail
          if (detailDto.id) {
            const existingDetail = existing.program_details.find(
              (d) => d.id === Number(detailDto.id),
            );

            if (existingDetail) {
              Object.assign(existingDetail, detailDto, {
                updated_by: updatedBy,
                updated_at: new Date(),
              });

              if (imageFile) {
                existingDetail.image = 'programs/' + imageFile.filename;
              }

              await this.programDetailRepo.save(existingDetail);
            }
          }
          // Case 2: Add new detail
          else {
            const newDetail = this.programDetailRepo.create({
              ...detailDto,
              program_id: id,
              image: imageFile ? 'programs/' + imageFile.filename : null,
              created_by: updatedBy,
              created_at: new Date(),
            });
            await this.programDetailRepo.save(newDetail);
          }
        }
      }

      // --- Update program main data ---
      const { details, ...programData } = dto;
      Object.assign(existing, programData, {
        updated_by: updatedBy,
        updated_at: new Date(),
      });

      delete (existing as any).program_details;
      await this.programRepo.save(existing);

      // --- Fetch updated data ---
      const finalData = await this.programRepo.findOne({
        where: { id },
        relations: ['program_details'],
      });

      return {
        status: true,
        message: 'Program and details updated successfully',
        data: finalData,
      };
    } catch (error) {
      console.error('Update Error:', error);
      return {
        status: false,
        message: 'Failed to update program and details',
        error: error.message,
      };
    }
  }

  // 🔁 TOGGLE STATUS
  async toggleStatus(id: number) {
    try {
      const program = await this.programRepo.findOne({
        where: { id },
        relations: ['program_details'],
      });

      if (!program) {
        return {
          status: false,
          message: `Program with ID ${id} not found`,
          data: null,
        };
      }

      program.status = program.status === 1 ? 0 : 1;
      await this.programRepo.save(program);

      await this.programDetailRepo
        .createQueryBuilder()
        .update(ProgramDetail)
        .set({ status: program.status })
        .where('program_id = :id', { id })
        .execute();

      return {
        status: true,
        message: `Program and details ${program.status === 1 ? 'activated' : 'deactivated'} successfully`,
        data: program,
      };
    } catch (error) {
      return {
        status: false,
        message: 'Failed to toggle program status',
        data: error.message,
      };
    }
  }

  async softDelete(id: number) {
    try {
      const program = await this.programRepo.findOne({
        where: { id },
        relations: ['program_details'],
      });

      if (!program) {
        return {
          status: false,
          message: `Program with ID ${id} not found`,
          data: null,
        };
      }

      // If already soft deleted
      if (program.status === 2) {
        return {
          status: false,
          message: `Program with ID ${id} is already deleted`,
          data: null,
        };
      }

      // Step 1: Generate base deleted slug
      // const baseSlug = `${program.slug}-DL`;

      // Step 2: Check existing deleted slugs
      // const existingDeleted = await this.programRepo
      //   .createQueryBuilder('p')
      //   .where('p.slug LIKE :slug', { slug: `${baseSlug}%` })
      //   .getMany();

      // // Step 3: Determine unique new slug
      // let newSlug = baseSlug;
      // if (existingDeleted.length > 0) {
      //   newSlug = `${baseSlug}${existingDeleted.length}`;
      // }

      // Step 4: Update program status & slug
      program.status = 2;
      // program.slug = newSlug;
      await this.programRepo.save(program);

      // Step 5: Update related program details to status = 2
      await this.programDetailRepo
        .createQueryBuilder()
        .update(ProgramDetail)
        .set({ status: 2 })
        .where('program_id = :id', { id })
        .execute();

      return {
        status: true,
        message: `Program deleted successfully`,
        data: program,
      };
    } catch (error) {
      return {
        status: false,
        message: 'Failed to soft delete program',
        data: error.message,
      };
    }
  }
}
